"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteInquerito = exports.getInqueritoEvento = exports.getOneInquerito = exports.getAllInqueritos = exports.createRespostaInquerito = exports.createInquerito = void 0;
const express_validator_1 = require("express-validator");
const inquerito_1 = __importDefault(require("../models/inquerito"));
const mongoose_1 = __importDefault(require("mongoose"));
const evento_1 = __importDefault(require("../models/evento"));
const participante_1 = __importDefault(require("../models/participante"));
const resposta_1 = __importDefault(require("../models/resposta"));
const pergunta_1 = __importDefault(require("../models/pergunta"));
const createInquerito = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { nome, evento } = req.body;
        const verifyEvento = await evento_1.default.findOne({ _id: evento });
        if (!verifyEvento) {
            const error = new Error(`Este evento não existe! `);
            error.statusCode = 422;
            throw error;
        }
        const verifyInquerito = await inquerito_1.default.findOne({ evento, nome });
        if (verifyInquerito) {
            const error = new Error(`Já existe um inquérito de ${verifyInquerito.nome} para este evento! `);
            error.statusCode = 409;
            throw error;
        }
        const inquerito = new inquerito_1.default({
            evento,
            nome
        });
        if (!verifyEvento.inqueritos.includes(inquerito._id)) {
            verifyEvento.inqueritos.push(inquerito._id);
            await verifyEvento.save(); // Salvar as mudanças no Evento
        }
        await inquerito.save();
        res.status(201).json({
            msg: 'Inquérito criado com sucesso',
            inquerito
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createInquerito = createInquerito;
const createRespostaInquerito = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { nome, bi, email, genero, telefone1, telefone2, categoria, presenca, provincia, municipio, rua, inquerito, perguntas } = req.body;
        if (telefone1 === telefone2) {
            const error = new Error(`Os números de telefone não devem iguais!`);
            error.statusCode = 409;
            throw error;
        }
        const verifyInquerito = await inquerito_1.default.findOne({ _id: inquerito });
        if (!verifyInquerito) {
            const error = new Error(`Este inquérito não existe!`);
            error.statusCode = 404;
            throw error;
        }
        const verifyEmail = await participante_1.default.findOne({ email, inquerito: verifyInquerito._id });
        const verifyBI = await participante_1.default.findOne({ bi, inquerito: verifyInquerito._id });
        const verifyTelefone = await participante_1.default.findOne({ telefone1, inquerito: verifyInquerito._id });
        const participante = new participante_1.default({
            nome,
            bi,
            email,
            genero,
            telefone1,
            telefone2,
            categoria,
            presenca,
            provincia,
            municipio,
            rua,
            inquerito
        });
        if (verifyBI) {
            const error = new Error(`Alguém já participou deste inquérito com este BI.`);
            error.statusCode = 409;
            throw error;
        }
        else if (verifyEmail) {
            const error = new Error(`Alguém já participou deste inquérito com este email.`);
            error.statusCode = 409;
            throw error;
        }
        else if (verifyTelefone) {
            const error = new Error(`Alguém já participou deste inquérito com este telefone.`);
            error.statusCode = 409;
            throw error;
        }
        // Usando for...of para tratar as operações assíncronas
        for (const { _id, resposta } of perguntas) {
            // Verificar se a pergunta pertence a este inquérito
            const verifyPergunta = await pergunta_1.default.findOne({ _id, inquerito: verifyInquerito._id });
            if (!verifyPergunta) {
                const error = new Error(`A pergunta não pertence a este inquérito!`);
                error.statusCode = 404;
                throw error;
            }
            // Verificar se já existe uma resposta do emissor para esta pergunta
            const verifyResposta = await resposta_1.default.findOne({
                emissor: participante._id, // O emissor da resposta
                pergunta: verifyPergunta._id // A pergunta verificada
            });
            if (verifyResposta) {
                return res.status(409).json({
                    success: false,
                    message: `Já respondeste à este inquérito.`
                });
            }
            else {
                if (!verifyInquerito.participantes.includes(participante._id)) {
                    verifyInquerito.participantes.push(participante._id);
                    console.log(verifyInquerito);
                    await verifyInquerito.save(); // Salvar as mudanças
                }
                await participante.save();
                const newResposta = new resposta_1.default({
                    texto: resposta,
                    pergunta: _id,
                    emissor: participante._id
                });
                verifyPergunta.respostas.push(newResposta._id);
                // Salvar resposta e pergunta
                await newResposta.save();
                await verifyPergunta.save();
            }
        }
        // Apenas uma resposta ao final
        res.status(201).json({
            success: true,
            participante
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createRespostaInquerito = createRespostaInquerito;
const getAllInqueritos = async (req, res, next) => {
    try {
        const inquerito = await inquerito_1.default.find({})
            .populate({ path: 'evento', populate: ('titulo') })
            .populate({
            path: 'perguntas',
            populate: {
                path: 'respostas',
                populate: {
                    path: 'emissor',
                }
            }
        })
            .exec();
        if (!inquerito) {
            return res.status(404).json({
                success: false,
                inquerito
            });
        }
        res.status(200).json({
            success: true,
            inquerito
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getAllInqueritos = getAllInqueritos;
const getOneInquerito = async (req, res, next) => {
    try {
        const id = req.params.id;
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (validId) {
            const inquerito = await inquerito_1.default.findById(id)
                .populate('evento')
                .populate({
                path: 'perguntas',
                populate: {
                    path: 'respostas',
                    populate: {
                        path: 'emissor',
                    }
                }
            })
                .populate('participantes')
                .exec();
            if (!inquerito) {
                return res.status(404).json({
                    success: false,
                    inquerito
                });
            }
            res.status(200).json({
                success: true,
                inquerito
            });
        }
        else {
            return res.status(400).json({
                success: false,
                message: 'Id inválido.'
            });
        }
    }
    catch (err) {
        next(err);
    }
};
exports.getOneInquerito = getOneInquerito;
const getInqueritoEvento = async (req, res, next) => {
    try {
        const { nome, tituloEvento } = req.params;
        const evento = await evento_1.default.findOne({ titulo: tituloEvento });
        if (!evento) {
            const error = new Error(`Não existe um evento com este título! `);
            error.statusCode = 422;
            throw error;
        }
        const verifyInquerito = await inquerito_1.default.findOne({ nome });
        if (!verifyInquerito) {
            const error = new Error(`Não existe um inquérito com este nome! `);
            error.statusCode = 422;
            throw error;
        }
        const inquerito = await inquerito_1.default.findOne({ evento: evento._id, nome })
            .populate('evento')
            .populate({
            path: 'perguntas',
            populate: {
                path: 'respostas',
                populate: {
                    path: 'emissor',
                },
            }
        })
            .populate('participantes')
            .exec();
        if (!inquerito) {
            throw new Error(`Inquérito com nome "${nome}" não encontrado para o evento "${tituloEvento}".`);
        }
        res.status(200).json({
            success: true,
            inquerito
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getInqueritoEvento = getInqueritoEvento;
const deleteInquerito = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const id = req.params.id;
        // Verifica se o id é válido
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (!validId) {
            return res.status(400).json({
                success: false,
                message: 'ID inválido.',
            });
        }
        // Busca o Inquerito para garantir que ele existe
        const inquerito = await inquerito_1.default.findById(id);
        if (!inquerito) {
            return res.status(404).json({
                success: false,
                message: 'Inquerito não encontrada.',
            });
        }
        await evento_1.default.updateMany({ inqueritos: inquerito._id }, { $pull: { inqueritos: inquerito._id } });
        await inquerito_1.default.findByIdAndDelete(id);
        res.status(200).json({
            success: true,
            message: `Inquérito deletado com sucesso.`,
        });
    }
    catch (err) {
        next(err); // Passa o erro para o middleware de erro
    }
};
exports.deleteInquerito = deleteInquerito;
