"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deletePainelEvento = exports.adicionarTemaPainelEvento = exports.getOnePainelEvento = exports.getAllPainelEventos = exports.createPainelEvento = void 0;
const express_validator_1 = require("express-validator");
const painel_evento_1 = __importDefault(require("../models/painel-evento"));
const mongoose_1 = __importDefault(require("mongoose"));
const evento_1 = __importDefault(require("../models/evento"));
const painel_1 = __importDefault(require("../models/painel"));
const tema_1 = __importDefault(require("../models/tema"));
const createPainelEvento = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { evento, painel } = req.body;
        const verifyEvento = await evento_1.default.findOne({ _id: evento });
        if (!verifyEvento) {
            const error = new Error(`Este evento não existe! `);
            error.statusCode = 422;
            throw error;
        }
        const verifyPainel = await painel_1.default.findOne({ _id: painel });
        if (!verifyPainel) {
            const error = new Error(`Este painel não existe! `);
            error.statusCode = 422;
            throw error;
        }
        const verifyPainelEvento = await painel_evento_1.default.findOne({ evento, painel });
        if (verifyPainelEvento) {
            const error = new Error(`Painel já adicionado no evento!`);
            error.statusCode = 422;
            throw error;
        }
        const painelEvento = new painel_evento_1.default({
            evento,
            painel
        });
        // Adicionar o painel ao array paineisEvento do Evento se não estiver presente
        if (!verifyEvento.paineisEvento.includes(painelEvento._id)) {
            verifyEvento.paineisEvento.push(painelEvento._id);
            await verifyEvento.save(); // Salvar as mudanças no Evento
        }
        await painelEvento.save();
        res.status(201).json({
            msg: 'Painel adicionado no evento com sucesso',
            painelEvento
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createPainelEvento = createPainelEvento;
const getAllPainelEventos = async (req, res, next) => {
    try {
        try {
            const paineisEvento = await painel_evento_1.default.find({})
                .populate({ path: 'evento' })
                .populate({ path: 'painel' })
                .populate({ path: 'temas' });
            if (paineisEvento.length === 0) {
                return res.status(404).json({
                    success: false,
                    paineisEvento
                });
            }
            res.status(200).json({
                success: true,
                paineisEvento
            });
        }
        catch (err) {
            next(err);
        }
    }
    catch (err) {
        next(err);
    }
};
exports.getAllPainelEventos = getAllPainelEventos;
const getOnePainelEvento = async (req, res, next) => {
    try {
        const id = req.params.id;
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (validId) {
            const paineisEvento = await painel_evento_1.default.find({});
            if (!paineisEvento) {
                return res.status(404).json({
                    success: false,
                    paineisEvento
                });
            }
            res.status(200).json({
                success: true,
                paineisEvento
            });
        }
        else {
            return res.status(400).json({
                success: false,
                message: 'Id inválido.'
            });
        }
    }
    catch (err) {
        next(err);
    }
};
exports.getOnePainelEvento = getOnePainelEvento;
const adicionarTemaPainelEvento = async (req, res, next) => {
    try {
        const { evento, painel, id_tema } = req.body;
        // Validar os IDs
        if (!mongoose_1.default.Types.ObjectId.isValid(evento) ||
            !mongoose_1.default.Types.ObjectId.isValid(painel) ||
            !mongoose_1.default.Types.ObjectId.isValid(id_tema)) {
            const error = new Error('ID(s) inválido(s)');
            error.statusCode = 400;
            throw error;
        }
        // Verificar se o Evento existe
        const verifyEvento = await evento_1.default.findById({ id: evento });
        if (!verifyEvento) {
            const error = new Error('Evento não encontrado');
            error.statusCode = 404;
            throw error;
        }
        // Verificar se o Tema existe
        const tema = await tema_1.default.findById(id_tema);
        if (!tema) {
            const error = new Error('Tema não encontrado');
            error.statusCode = 404;
            throw error;
        }
        // Encontrar ou criar o PainelEvento
        let painelEvento = await painel_evento_1.default.findOne({ evento, painel });
        if (!painelEvento) {
            painelEvento = new painel_evento_1.default({
                evento: evento,
                painel: painel,
                temas: []
            });
        }
        // Adicionar o painel ao array paineisEvento do Evento se não estiver presente
        if (!verifyEvento.paineisEvento.includes(painelEvento._id)) {
            verifyEvento.paineisEvento.push(painelEvento._id);
            await verifyEvento.save(); // Salvar as mudanças no Evento
        }
        // Adicionar o Tema se ainda não estiver presente
        if (!painelEvento.temas.includes(id_tema)) {
            painelEvento.temas.push(id_tema);
            await painelEvento.save();
            res.status(200).json({
                message: 'Tema adicionado com sucesso ao painel do evento',
                painelEvento
            });
        }
        else {
            res.status(200).json({
                message: 'Tema já adicionado no painel do evento',
                painelEvento
            });
        }
    }
    catch (err) {
        next(err);
    }
};
exports.adicionarTemaPainelEvento = adicionarTemaPainelEvento;
const deletePainelEvento = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const id = req.params.id;
        // Verifica se o id é válido
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (!validId) {
            return res.status(400).json({
                success: false,
                message: 'ID inválido.',
            });
        }
        // Busca o painel para garantir que ele existe
        const painelEvento = await painel_evento_1.default.findById(id);
        if (!painelEvento) {
            return res.status(404).json({
                success: false,
                message: 'Painel não encontrado no evento.',
            });
        }
        // Deleta o painelEvento
        await painel_evento_1.default.findByIdAndDelete(id);
        res.status(200).json({
            success: true,
            message: `Painel deletado do evento com sucesso.`,
        });
    }
    catch (err) {
        next(err); // Passa o erro para o middleware de erro
    }
};
exports.deletePainelEvento = deletePainelEvento;
