"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteSiteInfo = exports.changeImage = exports.editSiteInfo = exports.getSiteInfo = exports.getSiteInfos = exports.createSiteInfo = void 0;
const site_info_1 = __importDefault(require("../models/site-info"));
const express_validator_1 = require("express-validator");
const file_1 = require("../util/file");
const aws_1 = require("../util/aws");
const user_1 = __importDefault(require("../models/user"));
const pagination_1 = require("../util/pagination");
const main_1 = require("../util/main");
// Controller para criar siteInfo
const createSiteInfo = async (req, res, next) => {
    try {
        let { menu, opcao, conteudo, userId, } = req.body;
        const imagem = req.file;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const siteInfo = new site_info_1.default({
            menu,
            opcao,
            conteudo,
            admin: userId,
        });
        if (imagem) {
            imagem.filename = (0, file_1.generateFileName)(imagem);
            siteInfo.imagem = imagem.filename;
            await (0, aws_1.uploadFile)(imagem);
        }
        await siteInfo.save();
        // HistoricoActividade
        await (0, main_1.createHistoricoActividade)(`Criou siteInfo`, 'SiteInfo', siteInfo.id, admin);
        if (imagem) {
            siteInfo.imagem = await (0, aws_1.getURL)(siteInfo.imagem);
        }
        res.status(201).json({ msg: 'Sucesso', siteInfo });
    }
    catch (err) {
        next(err);
    }
};
exports.createSiteInfo = createSiteInfo;
// Metodo que busca todas as siteInfos
const getSiteInfos = async (req, res, next) => {
    try {
        const { status } = req.query;
        const filterObj = {};
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        if (Boolean(status)) {
            filterObj.status = status;
        }
        const total = await site_info_1.default.find(filterObj).count();
        const siteInfos = await site_info_1.default.find(filterObj)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage);
        for (const siteInfo of siteInfos) {
            if (siteInfo.imagem) {
                siteInfo.imagem = await (0, aws_1.getURL)(siteInfo.imagem);
            }
        }
        res.status(200).json({
            msg: 'Sucesso',
            siteInfos,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getSiteInfos = getSiteInfos;
// Metodo que busca pelo id siteInfo
const getSiteInfo = async (req, res, next) => {
    try {
        const { id } = req.params;
        const siteInfo = await site_info_1.default.findById(id)
            .populate({
            path: 'admin',
            populate: 'dadosPessoais'
        });
        if (!siteInfo) {
            const error = new Error('SiteInfo não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (siteInfo.imagem) {
            siteInfo.imagem = await (0, aws_1.getURL)(siteInfo.imagem);
        }
        if (siteInfo.admin.dadosPessoais) {
            siteInfo.admin.dadosPessoais.fotoURL = await (0, aws_1.getURL)(siteInfo.admin.dadosPessoais.fotoURL);
        }
        res.status(200).json({
            msg: 'Sucesso',
            siteInfo,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getSiteInfo = getSiteInfo;
// Controller para editar siteInfo
const editSiteInfo = async (req, res, next) => {
    try {
        let { siteInfoId, menu, opcao, conteudo, status, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const siteInfo = await site_info_1.default.findById(siteInfoId);
        if (!siteInfo) {
            const error = new Error('SiteInfo não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (Boolean(menu)) {
            siteInfo.menu = menu;
        }
        if (Boolean(opcao)) {
            siteInfo.opcao = opcao;
        }
        if (Boolean(conteudo)) {
            siteInfo.conteudo = conteudo;
        }
        if (Boolean(status)) {
            siteInfo.status = status;
        }
        await siteInfo.save();
        // HistoricoActividade
        await (0, main_1.createHistoricoActividade)(`Editou siteInfo`, 'SiteInfo', siteInfo.id, admin);
        if (siteInfo.imagem) {
            siteInfo.imagem = await (0, aws_1.getURL)(siteInfo.imagem);
        }
        res.status(201).json({ msg: 'Sucesso', siteInfo });
    }
    catch (err) {
        next(err);
    }
};
exports.editSiteInfo = editSiteInfo;
const changeImage = async (req, res, next) => {
    try {
        const { siteInfoId, userId } = req.body;
        const imagem = req.file;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        if (!imagem) {
            const error = new Error('Por favor fornceça uma imagem');
            error.statusCode = 422;
            throw error;
        }
        const siteInfo = await site_info_1.default.findById(siteInfoId);
        if (!siteInfo) {
            const error = new Error('SiteInfo não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        // Primeiro deleta o arquivo
        await (0, aws_1.deleteFile)(siteInfo.imagem);
        // Depois renomeia e faz o upload
        imagem.filename = (0, file_1.generateFileName)(imagem);
        await (0, aws_1.uploadFile)(imagem);
        siteInfo.imagem = imagem.filename;
        await siteInfo.save();
        // HistoricoActividade
        await (0, main_1.createHistoricoActividade)(`Alterou imagem do siteInfo`, 'SiteInfo', siteInfo.id, admin);
        siteInfo.imagem = await (0, aws_1.getURL)(siteInfo.imagem);
        res.status(201).json({
            msg: 'Sucesso!',
            siteInfo,
        });
    }
    catch (error) {
        next(error);
    }
};
exports.changeImage = changeImage;
const deleteSiteInfo = async (req, res, next) => {
    try {
        const { id, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const user = await user_1.default.findOne({ _id: userId, isAdmin: true });
        if (!user) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const siteInfo = await site_info_1.default.findById(id);
        if (!siteInfo) {
            const error = new Error('SiteInfo não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        await siteInfo.delete();
        await (0, main_1.createHistoricoActividade)(`Deletou siteInfo: ${siteInfo.menu}`, 'SiteInfo', siteInfo.id, user);
        res.status(204).json({ msg: 'Sucesso!' });
    }
    catch (err) {
        next(err);
    }
};
exports.deleteSiteInfo = deleteSiteInfo;
