"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteComunicado = exports.changeStatus = exports.editComunicado = exports.getComunicado = exports.getComunicados = exports.createComunicado = void 0;
const comunicado_1 = require("../models/comunicado");
const comunicado_2 = __importDefault(require("../models/comunicado"));
const express_validator_1 = require("express-validator");
const user_1 = __importDefault(require("../models/user"));
const pagination_1 = require("../util/pagination");
// Funcao Para registar Comunicado
const createComunicado = async (req, res, next) => {
    try {
        let { titulo, conteudo, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const user = await user_1.default.findById(userId);
        if (!user) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const comunicado = new comunicado_2.default({
            titulo,
            conteudo,
            admin: userId
        });
        await comunicado.save();
        res.status(201).json({ msg: 'Sucesso!', comunicado });
    }
    catch (err) {
        next(err);
    }
};
exports.createComunicado = createComunicado;
// Funcao Para pegar todos os Comunicados
const getComunicados = async (req, res, next) => {
    try {
        const { status } = req.query;
        const filterObj = {};
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        if (Boolean(status)) {
            filterObj.status = status;
        }
        const total = await comunicado_2.default.find(filterObj).count();
        const comunicados = await comunicado_2.default.find(filterObj)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage)
            .sort({ createdAt: -1 });
        res.status(200).json({
            msg: 'Sucesso',
            comunicados,
            ...(0, pagination_1.paginationResult)(total, page, perPage)
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getComunicados = getComunicados;
// Funcao que pega comunicado pelo id
const getComunicado = async (req, res, next) => {
    try {
        const { id } = req.params;
        const comunicado = await comunicado_2.default.findById(id);
        if (!comunicado) {
            const error = new Error('Comunicado não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        res.status(200).json({ msg: "Sucesso", comunicado });
    }
    catch (err) {
        next(err);
    }
};
exports.getComunicado = getComunicado;
// Metodo que actualiza a Comunicado
const editComunicado = async (req, res, next) => {
    try {
        let { comunicadoId, titulo, conteudo, status, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const comunicado = await comunicado_2.default.findById(comunicadoId);
        if (!comunicado) {
            const error = new Error('Comunicado não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (Boolean(status)) {
            if (!comunicado_1.comunicadoStatus.includes(status)) {
                const error = new Error('Status inválido!');
                error.statusCode = 422;
                throw error;
            }
            comunicado.status = status;
        }
        comunicado.titulo = titulo;
        comunicado.conteudo = conteudo;
        comunicado.editadoPor = userId;
        await comunicado.save();
        res.status(201).json({ msg: "Sucesso!", comunicado });
    }
    catch (err) {
        next(err);
    }
};
exports.editComunicado = editComunicado;
// Metodo para mudar status
const changeStatus = async (req, res, next) => {
    try {
        const { comunicadoId, status, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        if (!comunicado_1.comunicadoStatus.includes(status)) {
            const error = new Error('Status inválido!');
            error.statusCode = 422;
            throw error;
        }
        const user = await user_1.default.findById(userId);
        if (!user) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const comunicado = await comunicado_2.default.findById(comunicadoId);
        if (!comunicado) {
            const error = new Error('Comunicado não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        comunicado.status = status;
        comunicado.editadoPor = userId;
        await comunicado.save();
        res.status(201).json({ msg: 'Sucesso', comunicado });
    }
    catch (err) {
        next(err);
    }
};
exports.changeStatus = changeStatus;
// Metodo que deleta comunicado
const deleteComunicado = async (req, res, next) => {
    try {
        let { comunicadoId, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const user = await user_1.default.findById(userId);
        if (!user) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const comunicado = await comunicado_2.default.findById(comunicadoId);
        if (!comunicado) {
            const error = new Error('Comunicado não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        await comunicado.delete();
        res.status(204).json({ msg: "Sucesso!", comunicado });
    }
    catch (err) {
        next(err);
    }
};
exports.deleteComunicado = deleteComunicado;
