"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deletePermante = exports.editDadosAcademico = exports.getDadosAcademico = exports.getFormacoesAcademicasByMembro = exports.createDadosAcademico = void 0;
const file_1 = require("../util/file");
const express_validator_1 = require("express-validator");
const dados_academico_1 = __importDefault(require("../models/dados-academico"));
const aws_1 = require("../util/aws");
const pagination_1 = require("../util/pagination");
const membro_1 = __importDefault(require("../models/membro"));
// Metodo Para criar dadosAcademico
const createDadosAcademico = async (req, res, next) => {
    try {
        const { instituicao, tipo, cidade, pais, curso, mesInicio, anoInicio, mesFim, anoFim, nota, descricao, membroId, } = req.body;
        const anexo = req.file;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const membro = await membro_1.default.findById(membroId);
        if (!membro) {
            const error = new Error('Membro não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const dadosAcademico = new dados_academico_1.default({
            instituicao,
            tipo,
            mesInicio,
            anoInicio,
            mesFim,
            anoFim,
            membro: membroId,
        });
        if (Boolean(mesInicio)) {
            dadosAcademico.mesInicio = mesInicio;
        }
        if (Boolean(mesFim)) {
            dadosAcademico.mesFim = mesFim;
        }
        if (Boolean(curso)) {
            dadosAcademico.curso = curso;
        }
        if (Boolean(nota)) {
            dadosAcademico.nota = nota;
        }
        if (Boolean(descricao)) {
            dadosAcademico.descricao = descricao;
        }
        if (Boolean(cidade)) {
            dadosAcademico.cidade = cidade;
        }
        if (Boolean(pais)) {
            dadosAcademico.pais = pais;
        }
        if (anexo) {
            anexo.filename = (0, file_1.generateFileName)(anexo);
            dadosAcademico.anexo = anexo.filename;
        }
        await dadosAcademico.save();
        if (anexo) {
            await (0, aws_1.uploadFile)(anexo);
        }
        res.status(201).json({
            msg: 'Sucesso',
            dadosAcademico,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createDadosAcademico = createDadosAcademico;
// Metodo que busca todas formacoesAcademicas de um Membro
const getFormacoesAcademicasByMembro = async (req, res, next) => {
    try {
        const { membroId } = req.params;
        const filterObj = { membro: membroId };
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        const membro = await membro_1.default.findById(membroId);
        if (!membro) {
            const error = new Error('Membro não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const total = await dados_academico_1.default.find(filterObj).count();
        const formacoesAcademicas = await dados_academico_1.default.find(filterObj)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage)
            .sort({ anoFim: -1, mesFim: -1 });
        for (const dadosAcademico of formacoesAcademicas) {
            if (dadosAcademico.anexo)
                dadosAcademico.anexo = await (0, aws_1.getURL)(dadosAcademico.anexo);
        }
        res.status(200).json({
            msg: 'Sucesso',
            formacoesAcademicas,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getFormacoesAcademicasByMembro = getFormacoesAcademicasByMembro;
// Metodo que busca dadosAcademico pelo id
const getDadosAcademico = async (req, res, next) => {
    try {
        const { id } = req.params;
        const dadosAcademico = await dados_academico_1.default.findById(id);
        if (!dadosAcademico) {
            const error = new Error('Formação Acadêmica não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (dadosAcademico.anexo) {
            dadosAcademico.anexo = await (0, aws_1.getURL)(dadosAcademico.anexo);
        }
        res.status(200).json({
            msg: 'Sucesso',
            dadosAcademico: dadosAcademico,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getDadosAcademico = getDadosAcademico;
const editDadosAcademico = async (req, res, next) => {
    try {
        const { id, instituicao, tipo, curso, mesInicio, anoInicio, mesFim, anoFim, nota, descricao, cidade, pais, } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const dadosAcademico = await dados_academico_1.default.findById(id);
        if (!dadosAcademico) {
            const error = new Error('Formação Acadêmica não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (Boolean(instituicao)) {
            dadosAcademico.instituicao = instituicao;
        }
        if (Boolean(tipo)) {
            dadosAcademico.tipo = tipo;
        }
        if (Boolean(curso)) {
            dadosAcademico.curso = curso;
        }
        if (Boolean(mesInicio)) {
            dadosAcademico.mesInicio = mesInicio;
        }
        if (Boolean(anoInicio)) {
            dadosAcademico.anoInicio = anoInicio;
        }
        if (Boolean(mesFim)) {
            dadosAcademico.mesFim = mesFim;
        }
        if (Boolean(anoFim)) {
            dadosAcademico.anoFim = anoFim;
        }
        if (Boolean(nota)) {
            dadosAcademico.nota = nota;
        }
        if (Boolean(cidade)) {
            dadosAcademico.cidade = cidade;
        }
        if (Boolean(pais)) {
            dadosAcademico.pais = pais;
        }
        if (Boolean(descricao)) {
            dadosAcademico.descricao = descricao;
        }
        await dadosAcademico.save();
        if (dadosAcademico.anexo) {
            dadosAcademico.anexo = await (0, aws_1.getURL)(dadosAcademico.anexo);
        }
        res.status(201).json({
            msg: 'Sucesso',
            dadosAcademico,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.editDadosAcademico = editDadosAcademico;
const deletePermante = async (req, res, next) => {
    try {
        const { id } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const dadosAcademico = await dados_academico_1.default.findById(id);
        if (!dadosAcademico) {
            const error = new Error(`Formação Acadêmica não encontrada!`);
            error.statusCode = 422;
            throw error;
        }
        dadosAcademico.anexo && (await (0, aws_1.deleteFile)(dadosAcademico.anexo));
        await dados_academico_1.default.deleteOne({ _id: id });
        res.status(204).json({
            msg: 'Deletado com sucesso',
        });
    }
    catch (error) {
        next(error);
    }
};
exports.deletePermante = deletePermante;
